import React from 'react';
import {
  useCurrentFrame,
  useVideoConfig,
  interpolate,
  Easing,
  AbsoluteFill,
} from 'remotion';

// =============================================================================
// COMPOSITION CONFIG (Required for auto-discovery)
// =============================================================================

export const compositionConfig = {
  id: 'AICountdownMasterpiece',
  durationInSeconds: 10,
  fps: 30,
  width: 1920,
  height: 1080,
};

// =============================================================================
// PRE-GENERATED DATA
// =============================================================================

const PARTICLE_COUNT = 40;
const particles = Array.from({ length: PARTICLE_COUNT }).map((_, i) => ({
  id: i,
  x: Math.random() * 100,
  y: Math.random() * 100,
  size: Math.random() * 6 + 2,
  speed: Math.random() * 2 + 0.5,
  delay: Math.random() * 100,
}));

const gridLines = Array.from({ length: 20 }).map((_, i) => i);

// =============================================================================
// HELPER COMPONENTS
// =============================================================================

const Particle: React.FC<{
  x: number;
  y: number;
  size: number;
  speed: number;
  delay: number;
  frame: number;
}> = ({ x, y, size, speed, delay, frame }) => {
  const movement = interpolate(
    (frame + delay) % 300,
    [0, 300],
    [0, -100],
    { extrapolateRight: 'clamp' }
  );

  const opacity = interpolate(
    (frame + delay) % 300,
    [0, 50, 250, 300],
    [0, 0.6, 0.6, 0],
    { extrapolateRight: 'clamp' }
  );

  return (
    <div
      style={{
        position: 'absolute',
        left: `${x}%`,
        top: `${y}%`,
        width: size,
        height: size,
        backgroundColor: '#00f2ff',
        borderRadius: '50%',
        boxShadow: '0 0 10px #00f2ff',
        opacity,
        transform: `translateY(${movement}px)`,
      }}
    />
  );
};

// =============================================================================
// MAIN COMPONENT
// =============================================================================

const AICountdownMasterpiece: React.FC = () => {
  const frame = useCurrentFrame();
  const { fps, durationInFrames, width, height } = useVideoConfig();

  // Background Parallax & Pulse
  const bgScale = interpolate(frame, [0, durationInFrames], [1, 1.15]);
  const bgRotate = interpolate(frame, [0, durationInFrames], [0, 5]);

  // Entrance Animations
  const titleEntrance = interpolate(frame, [0, 30], [0, 1], {
    easing: Easing.out(Easing.back(1.5)),
    extrapolateRight: 'clamp',
  });

  const subtitleEntrance = interpolate(frame, [15, 45], [0, 1], {
    easing: Easing.out(Easing.quad),
    extrapolateRight: 'clamp',
  });

  // Floating effect for main UI
  const floatY = Math.sin(frame / 15) * 20;

  // Text Glow Pulse
  const glow = interpolate(
    Math.sin(frame / 10),
    [-1, 1],
    [10, 30]
  );

  return (
    <AbsoluteFill
      style={{
        backgroundColor: '#050505',
        overflow: 'hidden',
        fontFamily: 'Poppins, Roboto, Helvetica, Arial, sans-serif',
      }}
    >
      {/* BACKGROUND LAYER: Animated Grid */}
      <AbsoluteFill
        style={{
          transform: `scale(${bgScale}) rotate(${bgRotate}deg)`,
          opacity: 0.3,
        }}
      >
        {gridLines.map((i) => (
          <div
            key={`v-${i}`}
            style={{
              position: 'absolute',
              left: `${i * 5}%`,
              width: '1px',
              height: '100%',
              background: 'linear-gradient(to bottom, transparent, #00f2ff, transparent)',
            }}
          />
        ))}
        {gridLines.map((i) => (
          <div
            key={`h-${i}`}
            style={{
              position: 'absolute',
              top: `${i * 10}%`,
              width: '100%',
              height: '1px',
              background: 'linear-gradient(to right, transparent, #7000ff, transparent)',
            }}
          />
        ))}
      </AbsoluteFill>

      {/* MID LAYER: Particles */}
      {particles.map((p) => (
        <Particle key={p.id} {...p} frame={frame} />
      ))}

      {/* CONTENT LAYER */}
      <AbsoluteFill
        style={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          justifyContent: 'center',
          transform: `translateY(${floatY}px)`,
        }}
      >
        {/* Main Badge */}
        <div
          style={{
            opacity: titleEntrance,
            transform: `scale(${titleEntrance})`,
            padding: '10px 30px',
            background: 'rgba(112, 0, 255, 0.2)',
            border: '2px solid #7000ff',
            borderRadius: '50px',
            color: '#fff',
            fontSize: '24px',
            fontWeight: 'bold',
            letterSpacing: '4px',
            marginBottom: '40px',
            boxShadow: '0 0 20px rgba(112, 0, 255, 0.4)',
            textTransform: 'uppercase',
          }}
        >
          System Active
        </div>

        {/* Headline */}
        <h1
          style={{
            opacity: titleEntrance,
            transform: `translateY(${(1 - titleEntrance) * 50}px)`,
            margin: 0,
            fontSize: '120px',
            fontWeight: 900,
            color: 'white',
            textAlign: 'center',
            lineHeight: 1.1,
            textShadow: `0 0 ${glow}px rgba(0, 242, 255, 0.8)`,
            letterSpacing: '-2px',
          }}
        >
          FUTURE OF <br />
          <span style={{ 
            color: '#00f2ff', 
            background: 'linear-gradient(90deg, #00f2ff, #7000ff)',
            WebkitBackgroundClip: 'text',
            WebkitTextFillColor: 'transparent'
          }}>
            CREATIVITY
          </span>
        </h1>

        {/* Sub-headline */}
        <p
          style={{
            opacity: subtitleEntrance,
            transform: `translateY(${(1 - subtitleEntrance) * 30}px)`,
            marginTop: '30px',
            fontSize: '36px',
            color: '#aaa',
            maxWidth: '800px',
            textAlign: 'center',
            lineHeight: 1.5,
            fontWeight: 300,
          }}
        >
          Automated video generation powered by <br />
          <strong>AICONTROVERSY.COM</strong> engine.
        </p>

        {/* Decorative Progress Bar */}
        <div style={{
          marginTop: '60px',
          width: '400px',
          height: '6px',
          background: 'rgba(255,255,255,0.1)',
          borderRadius: '3px',
          overflow: 'hidden',
          opacity: subtitleEntrance
        }}>
          <div style={{
            width: `${interpolate(frame, [0, durationInFrames], [0, 100])}%`,
            height: '100%',
            background: 'linear-gradient(90deg, #00f2ff, #7000ff)',
            boxShadow: '0 0 15px #00f2ff'
          }} />
        </div>
      </AbsoluteFill>

      {/* FRAME OVERLAY (Vignette) */}
      <AbsoluteFill
        style={{
          boxShadow: 'inset 0 0 300px rgba(0,0,0,0.9)',
          pointerEvents: 'none',
        }}
      />

      {/* CORNER ELEMENTS */}
      <div style={{
        position: 'absolute',
        top: '60px',
        left: '60px',
        color: '#00f2ff',
        fontSize: '20px',
        fontWeight: 'bold',
        fontFamily: 'monospace',
        opacity: 0.6
      }}>
        REV_02.TSX // {frame.toString().padStart(4, '0')}
      </div>
      
      <div style={{
        position: 'absolute',
        bottom: '60px',
        right: '60px',
        width: '60px',
        height: '60px',
        borderRight: '4px solid #7000ff',
        borderBottom: '4px solid #7000ff',
        opacity: subtitleEntrance
      }} />
    </AbsoluteFill>
  );
};

export default AICountdownMasterpiece;